﻿'----------------------------------------------------------
' Sample program for DS102/DS112 controller
'
' Copyright 2023 SURUGA SEIKI Co.,Ltd. All rights reserved.
'----------------------------------------------------------

Public Class Form1

    Private AxisNo As String = "1"          'Axis number
    Private Direction As String             'Drive direction setting(-(CCW)、+(CW))
    Private Mode As Short = 0               'Drive mode (0: Continuous, 1: Step, 2: Origin)

    '''<summary>
    '''Initialization
    '''</summary>
    Private Sub Form1_Load(sender As Object, e As EventArgs) Handles MyBase.Load

        'Display Software version
        Dim asm As System.Reflection.Assembly = System.Reflection.Assembly.GetExecutingAssembly()
        Dim ver As System.Version = asm.GetName().Version
        Dim softwareVersion As System.Version = New Version(ver.Major, ver.Minor, ver.Build)
        Text = "Sample program for DS102/DS112 controller Ver." + softwareVersion.ToString

        ComboBoxOrg.SelectedIndex = 0
        ComboBoxCommPort.SelectedIndex = 0
        ComboBoxBaudrate.SelectedIndex = 0
        TextBoxPosition.Text = "0"

    End Sub

    '''<summary>
    '''When press the connect button
    '''</summary>
    Private Sub ButtonConnect_Click(sender As Object, e As EventArgs) Handles ButtonConnect.Click

        'Set COM port
        CommPortOpen()

    End Sub

    '''<summary>
    '''When press the connect button
    '''</summary>
    Private Sub CommPortOpen()

        If SerialPort.IsOpen = True Then
            SerialPort.Close()
        End If

        SerialPort.PortName = ComboBoxCommPort.Text                     'COM number
        SerialPort.BaudRate = Integer.Parse(ComboBoxBaudrate.Text)      'Baud rate
        SerialPort.NewLine = vbCr                                       'Delimiter code
        SerialPort.ReadTimeout = 2000                                   'Read timeout time
        SerialPort.WriteTimeout = 2000                                  'Write timeout time

        Try
            'COM port open
            SerialPort.Open()

        Catch ex As System.Exception

            LabelState.Text = "Connection error"
            MessageBox.Show(ex.Message)
            Exit Sub

        End Try

        System.Threading.Thread.Sleep(100)

        '---------------------------------------------------------
        ' Request ID - Check DS102 / DS112 controller connection
        '---------------------------------------------------------
        Dim data As String

        data = SendReceiveString("*IDN?")

        Dim ret As Boolean = If(Not String.IsNullOrEmpty(data), data.Contains("SURUGA,DS1"), False)

        If ret Then

            '---------------------------------------------------------
            'Request version
            '---------------------------------------------------------
            data = SendReceiveString("DS102VER?")
            LabelFirmware.Text = data

            '---------------------------------------------------------
            'Number of control axes
            '---------------------------------------------------------
            data = SendReceiveString("CONTA?")

            ButtonAxisX.Enabled = False
            ButtonAxisY.Enabled = False
            ButtonAxisZ.Enabled = False
            ButtonAxisU.Enabled = False
            ButtonAxisV.Enabled = False
            ButtonAxisW.Enabled = False

            If Integer.Parse(data) = 2 Then
                'Number of axes is 2(X,Y)
                '---------------------------------------------------------
                ' Setting unit and speed table
                '---------------------------------------------------------
                For axNo As Integer = 1 To 2

                    SendString("AXI" + axNo.ToString + ":UNIT 0:SELSP 0")
                    System.Threading.Thread.Sleep(100)
                Next

                ButtonAxisX.Enabled = True
                ButtonAxisY.Enabled = True

            ElseIf Integer.Parse(data) = 4 Then
                'Number of axes is 4(X-U)
                '---------------------------------------------------------
                ' Setting unit and speed table
                '---------------------------------------------------------
                For axNo As Integer = 1 To 4

                    SendString("AXI" + axNo.ToString + ":UNIT 0:SELSP 0")
                    System.Threading.Thread.Sleep(100)
                Next

                ButtonAxisX.Enabled = True
                ButtonAxisY.Enabled = True
                ButtonAxisZ.Enabled = True
                ButtonAxisU.Enabled = True

            ElseIf Integer.Parse(data) = 6 Then
                'Number of axes is 6(X-W)
                '---------------------------------------------------------
                ' Setting unit and speed table
                '---------------------------------------------------------
                For axNo As Integer = 1 To 6

                    SendString("AXI" + axNo.ToString + ":UNIT 0:SELSP 0")
                    System.Threading.Thread.Sleep(100)
                Next

                ButtonAxisX.Enabled = True
                ButtonAxisY.Enabled = True
                ButtonAxisZ.Enabled = True
                ButtonAxisU.Enabled = True
                ButtonAxisV.Enabled = True
                ButtonAxisW.Enabled = True
            End If

            UpdateStatus()

        Else
            LabelState.Text = "Receive error"
            MessageBox.Show("DS102/DS112 controller is not connected to " + ComboBoxCommPort.Text)

        End If

    End Sub

    '''<summary>
    '''When press the disconnect button
    '''</summary>
    Private Sub ButtonDisconnect_Click(sender As Object, e As EventArgs) Handles ButtonDisconnect.Click

        If SerialPort.IsOpen = True Then
            SerialPort.Close()
        End If

        ButtonAxisX.Enabled = True
        ButtonAxisY.Enabled = True
        ButtonAxisZ.Enabled = True
        ButtonAxisU.Enabled = True
        ButtonAxisV.Enabled = True
        ButtonAxisW.Enabled = True
        LabelState.Text = "Disconnected"

    End Sub

    '''<summary>
    '''When select the X button
    '''</summary>
    Private Sub ButtonAxisX_CheckedChanged(sender As Object, e As EventArgs) Handles ButtonAxisX.CheckedChanged
        AxisNo = "1"            'Set the active axis to the X-axis
        UpdateStatus()
    End Sub

    '''<summary>
    '''When select the Y button
    '''</summary>
    Private Sub ButtonAxisY_CheckedChanged(sender As Object, e As EventArgs) Handles ButtonAxisY.CheckedChanged
        AxisNo = "2"            'Set the active axis to the Y-axis
        UpdateStatus()
    End Sub

    '''<summary>
    '''When select the Z button
    '''</summary>
    Private Sub ButtonAxisZ_CheckedChanged(sender As Object, e As EventArgs) Handles ButtonAxisZ.CheckedChanged
        AxisNo = "3"            'Set the active axis to the Z-axis
        UpdateStatus()
    End Sub

    '''<summary>
    '''When select the U button
    '''</summary>
    Private Sub ButtonAxisU_CheckedChanged(sender As Object, e As EventArgs) Handles ButtonAxisU.CheckedChanged
        AxisNo = "4"            'Set the active axis to the U-axis
        UpdateStatus()
    End Sub

    '''<summary>
    '''When select the V button
    '''</summary>
    Private Sub ButtonAxisV_CheckedChanged(sender As Object, e As EventArgs) Handles ButtonAxisV.CheckedChanged
        AxisNo = "5"            'Set the active axis to the V-axis
        UpdateStatus()
    End Sub

    '''<summary>
    '''When select the W button
    '''</summary>
    Private Sub ButtonAxisW_CheckedChanged(sender As Object, e As EventArgs) Handles ButtonAxisW.CheckedChanged
        AxisNo = "6"            'Set the active axis to the W-axis
        UpdateStatus()
    End Sub

    '''<summary>
    '''When select the continue button
    '''</summary>
    Private Sub RadioButtonContinueMode_CheckedChanged(sender As Object, e As EventArgs) Handles RadioButtonContinueMode.CheckedChanged
        ButtonCCW.Text = "- (CCW)"
        ButtonCW.Text = "+ (CW)"
        Mode = 0
    End Sub

    '''<summary>
    '''When select the step button
    '''</summary>
    Private Sub RadioButtonStepMode_CheckedChanged(sender As Object, e As EventArgs) Handles RadioButtonStepMode.CheckedChanged
        ButtonCCW.Text = "- (CCW)"
        ButtonCW.Text = "+ (CW)"
        Mode = 1
    End Sub

    '''<summary>
    '''When select the origin button
    '''</summary>
    Private Sub RadioButtonOrgMode_CheckedChanged(sender As Object, e As EventArgs) Handles RadioButtonOrgMode.CheckedChanged
        ButtonCCW.Text = "Origin"
        ButtonCW.Text = "Origin"
        Mode = 2
    End Sub

    '''<summary>
    '''When press the stop button
    '''</summary>
    Private Sub ButtonStop_Click(sender As Object, e As EventArgs) Handles ButtonStop.Click
        '---------------------------------------------------------
        ' Stop
        '---------------------------------------------------------
        SendString("STOP 0")
    End Sub

    '''<summary>
    '''When press the CCW button
    '''</summary>
    Private Sub ButtonCCW_MouseDown(sender As Object, e As MouseEventArgs) Handles ButtonCCW.MouseDown
        Direction = "CCW"               'Set the motor drive direction to CCW
        MoveStage()                     'Start the drive
    End Sub

    '''<summary>
    '''When release the CCW button
    '''</summary>
    Private Sub ButtonCCW_MouseUp(sender As Object, e As MouseEventArgs) Handles ButtonCCW.MouseUp
        'In the case of continue drive mode
        If Mode = 0 Then
            '---------------------------------------------------------
            ' Stop
            '---------------------------------------------------------
            SendString("STOP 0")        'Stop the axis
        End If
    End Sub

    '''<summary>
    '''When press the CW button
    '''</summary>
    Private Sub ButtonCW_MouseDown(sender As Object, e As MouseEventArgs) Handles ButtonCW.MouseDown
        Direction = "CW"                'Set the motor drive direction to CW
        MoveStage()                     'Start the drive
    End Sub

    '''<summary>
    '''When release the CW button
    '''</summary>
    Private Sub ButtonCW_MouseUp(sender As Object, e As MouseEventArgs) Handles ButtonCW.MouseUp
        'In the case of continue drive mode
        If Mode = 0 Then
            '---------------------------------------------------------
            ' Stop
            '---------------------------------------------------------
            SendString("STOP 0")        'Stop the axis
        End If
    End Sub

    '''<summary>
    '''Drive the stage
    '''</summary>
    Public Sub MoveStage()

        'Drive mode
        Select Case Mode
            Case 0
                'Continue
                'If the mortor drive direction is CW
                If Direction = "CW" Then
                    '---------------------------------------------------------
                    ' Continue drive / CW direction
                    '---------------------------------------------------------
                    SendString("AXI" + AxisNo + ":L0 " + TextBoxLSpeed.Text +
                               ":R0 " + TextBoxRate.Text + ":S0 " + TextBoxSRate.Text +
                               ":F0 " + TextBoxSpeed.Text + ":GO CWJ")
                    'If the mortor drive direction is CCW
                Else
                    '---------------------------------------------------------
                    ' Continue drive / CCW direction
                    '---------------------------------------------------------
                    SendString("AXI" + AxisNo + ":L0 " + TextBoxLSpeed.Text +
                               ":R0 " + TextBoxRate.Text + ":S0 " + TextBoxSRate.Text +
                               ":F0 " + TextBoxSpeed.Text + ":GO CCWJ")
                End If
            Case 1
                'Step
                '---------------------------------------------------------
                ' Step drive
                '---------------------------------------------------------
                SendString("AXI" + AxisNo + ":L0 " + TextBoxLSpeed.Text +
                           ":R0 " + TextBoxRate.Text + ":S0 " + TextBoxSRate.Text +
                           ":F0 " + TextBoxSpeed.Text + ":PULS " + TextBoxStep.Text + ":GO " + Direction)
            Case 2
                'Origin
                'Get the origin type
                Dim orgMode As String = ComboBoxOrg.SelectedIndex.ToString()

                'Change the origin type
                '---------------------------------------------------------
                ' Memory switch 0 setting
                '---------------------------------------------------------
                SendString("AXI" + AxisNo + ":MEMSW0 " + orgMode)

                System.Threading.Thread.Sleep(100)

                '---------------------------------------------------------
                ' Origin drive
                '---------------------------------------------------------
                SendString("AXI" + AxisNo + ":L0 " + TextBoxLSpeed.Text +
                           ":R0 " + TextBoxRate.Text + ":S0 " + TextBoxSRate.Text +
                           ":F0 " + TextBoxSpeed.Text + ":GO ORG")
        End Select

        Timer.Start()

    End Sub

    '''<summary>
    '''Screen update
    '''</summary>
    Private Sub Timer_Tick(sender As Object, e As EventArgs) Handles Timer.Tick

        If SerialPort.IsOpen = True Then
            UpdateStatus()
        End If

    End Sub

    '''<summary>
    '''Status update
    '''</summary>
    Public Sub UpdateStatus()

        Dim data As String
        Dim result As Integer

        '---------------------------------------------------------
        ' Request status binary 3
        '---------------------------------------------------------
        data = SendReceiveString("AXI" + AxisNo + ":SB3?")

        'Terminates if it cannot be converted to a numerical value
        If Not Integer.TryParse(data, result) Then
            Timer.Stop()
            Exit Sub
        End If

        'If axis selection is not available
        If (Integer.Parse(data) And &H1) <> &H1 Then
            LabelState.Text = "Axes cannot be selected"
            Timer.Stop()
        Else
            '---------------------------------------------------------
            ' Request status binary 1
            '---------------------------------------------------------
            data = SendReceiveString("AXI" + AxisNo + ":SB1?")

            If (Integer.Parse(data) And &H40) = &H40 Then
                LabelState.Text = "Driving"

            ElseIf (Integer.Parse(data) And &H10) = &H10 Then
                LabelState.Text = "Detect origin"
                Timer.Stop()

            ElseIf (Integer.Parse(data) And &H2) = &H2 Or (Integer.Parse(data) And &H4) = &H4 Then
                'Detect limit
                '---------------------------------------------------------
                ' Request status binary 2
                '---------------------------------------------------------
                data = SendReceiveString("AXI" + AxisNo + ":SB2?")

                If (Integer.Parse(data) And &H3) = &H3 Then
                    LabelState.Text = "Stage not connected"

                ElseIf (Integer.Parse(data) And &H1) = &H1 Then
                    LabelState.Text = "Detect CW limit"

                ElseIf (Integer.Parse(data) And &H2) = &H2 Then
                    LabelState.Text = "Detect CCW limit"

                ElseIf (Integer.Parse(data) And &H4) = &H4 Then
                    LabelState.Text = "Detect CW software limit"

                ElseIf (Integer.Parse(data) And &H8) = &H8 Then
                    LabelState.Text = "Detect CCW software limit"
                End If

                Timer.Stop()

            Else

                LabelState.Text = "Stop"
                Timer.Stop()

            End If

            '---------------------------------------------------------
            ' Request the current position
            '---------------------------------------------------------
            data = SendReceiveString("AXI" + AxisNo + ":POS?")
            TextBoxPosition.Text = data

        End If

    End Sub

    '''<summary>
    '''Send
    '''</summary>
    '''<param name="cmd">Send string</param>
    Public Sub SendString(cmd As String)

        Try
            If SerialPort.IsOpen Then
                'Send
                SerialPort.WriteLine(cmd)
            End If

        Catch ex As System.Exception
            LabelState.Text = "Send error"
            MessageBox.Show(ex.Message)

        End Try

    End Sub

    '''<summary>
    '''Receive
    '''</summary>
    '''<returns>Receive string</returns>
    Public Function ReceiveString() As String

        Dim timeoutTime As Date = Date.Now.AddMilliseconds(2000)        'Timeout time
        Dim rcvData As String = Nothing                                 'Receive data

        'Perform receiving process until the delimiter code
        While True

            Try

                Dim len As Integer = SerialPort.BytesToRead

                'Attempt receiving process until timeout
                While len < 1

                    System.Threading.Thread.Sleep(50)

                    If Date.Now > timeoutTime Then
                        LabelState.Text = "Timeout error"
                        Return Nothing
                    End If

                    len = SerialPort.BytesToRead

                End While

                Dim readBytes As Integer = 0
                Dim inByte As Byte() = New Byte(len) {}

                'Read data from the received buffer
                SerialPort.Read(inByte, readBytes, len)
                'Store the the received data in a string
                Dim rcvDataTemp As String = System.Text.Encoding.ASCII.GetString(inByte)

                rcvData += rcvDataTemp

                'Confirm the position of the delimiter code
                Dim index As Integer = rcvData.IndexOf(SerialPort.NewLine)

                'When a delimiter code is found
                If index > -1 Then
                    'Remove the delimiter code
                    Dim rcvDataReturn As String = rcvData.Remove(index)
                    Return rcvDataReturn
                End If

            Catch ex As System.Exception

                LabelState.Text = "Receive error"
                MessageBox.Show(ex.Message)
                Return Nothing

            End Try

        End While

        Return Nothing

    End Function

    '''<summary>
    '''Send and receive
    '''</summary>
    '''<param name="cmd">Send string</param>
    '''<returns>Receive string</returns>
    Public Function SendReceiveString(cmd As String) As String

        Dim data As String = Nothing

        If SerialPort.IsOpen = True Then

            'Send
            SendString(cmd)

            System.Threading.Thread.Sleep(100)

            'Receive
            data = ReceiveString()
        End If

        Return data

    End Function

    '''<summary>
    '''When press the position set button
    '''</summary>
    Private Sub ButtonSetPosition_Click(sender As Object, e As EventArgs) Handles ButtonSetPosition.Click
        '---------------------------------------------------------
        ' Set the current position
        '---------------------------------------------------------
        SendString("AXI" + AxisNo + ":POS " + TextBoxPosition.Text)
    End Sub

    '''<summary>
    '''When press the close button
    '''</summary>
    Private Sub ButtonClose_Click(sender As Object, e As EventArgs) Handles ButtonClose.Click
        Close()
    End Sub

    '''<summary>
    '''End-of-process handling
    '''</summary>
    Private Sub Form1_FormClosing(sender As Object, e As FormClosingEventArgs) Handles MyBase.FormClosing
        If SerialPort.IsOpen = True Then
            SerialPort.Close()
        End If
    End Sub

End Class
