﻿//----------------------------------------------------------
// Sample program for DS102/DS112 controller
//
// Copyright 2023 SURUGA SEIKI Co.,Ltd. All rights reserved.
//----------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace DS102CSharpSample
{
    public partial class Form1 : Form
    {
        private string AxisNo = "1";        //Axis number
        private string Direction;           //Drive direction setting(-(CCW)、+(CW))
        private int Mode = 0;               //Drive mode (0: Continuous, 1: Step, 2: Origin)

        ///<summary>
        ///Initialization
        ///</summary>
        public Form1()
        {
            InitializeComponent();

            //Display Software version
            System.Reflection.Assembly asm = System.Reflection.Assembly.GetExecutingAssembly();
            Version ver = asm.GetName().Version;
            Version softwareVersion = new Version(ver.Major, ver.Minor, ver.Build);

            this.Text = "Sample program for DS102/DS112 controller Ver." + softwareVersion;

            ComboBoxOrg.SelectedIndex = 0;
            ComboBoxCommPort.SelectedIndex = 0;
            ComboBoxBaudrate.SelectedIndex = 0;
            TextBoxPosition.Text = "0";
        }

        ///<summary>
        ///When press the connect button
        ///</summary>
        private void ButtonConnect_Click(object sender, EventArgs e)
        {
            //Set COM port
            CommPortOpen();
        }

        ///<summary>
        ///Set COM port
        ///</summary>
        private void CommPortOpen()
        {
            if (SerialPort != null)
                SerialPort.Close();

            SerialPort.PortName = ComboBoxCommPort.Text;                //COM number
            SerialPort.BaudRate = int.Parse(ComboBoxBaudrate.Text);     //Baud rate
            SerialPort.NewLine = "\r";                                  //Delimiter code
            SerialPort.ReadTimeout = 2000;                              //Read timeout time
            SerialPort.WriteTimeout = 2000;                             //Write timeout time

            try
            {
                //COM port open
                SerialPort.Open();
            }
            catch (Exception ex)
            {
                LabelState.Text = "Connection error";
                MessageBox.Show(ex.Message);
                return;
            }

            System.Threading.Thread.Sleep(100);

            //---------------------------------------------------------
            // Request ID - Check DS102 / DS112 controller connection
            //---------------------------------------------------------
            string data = SendReceiveString("*IDN?");

            if (!string.IsNullOrEmpty(data) ? data.Contains("SURUGA,DS1") : false)
            {
                //---------------------------------------------------------
                // Request version
                //---------------------------------------------------------
                data = SendReceiveString("DS102VER?");
                LabelFirmware.Text = data;

                //---------------------------------------------------------
                //Number of control axes
                //---------------------------------------------------------
                data = SendReceiveString("CONTA?");

                ButtonAxisX.Enabled = false;
                ButtonAxisY.Enabled = false;
                ButtonAxisZ.Enabled = false;
                ButtonAxisU.Enabled = false;
                ButtonAxisV.Enabled = false;
                ButtonAxisW.Enabled = false;

                //Number of axes is 2(X,Y)
                if (int.Parse(data) == 2)
                {
                    //---------------------------------------------------------
                    // Setting unit and speed table
                    //---------------------------------------------------------
                    for (int axNo = 1; axNo <= 2; axNo++)
                    {
                        SendString("AXI" + axNo.ToString() + ":UNIT 0:SELSP 0");
                        System.Threading.Thread.Sleep(100);
                    }

                    ButtonAxisX.Enabled = true;
                    ButtonAxisY.Enabled = true;
                }
                //Number of axes is 4(X-U)
                else if (int.Parse(data) == 4)
                {
                    //---------------------------------------------------------
                    // Setting unit and speed table
                    //---------------------------------------------------------
                    for (int axNo = 1; axNo <= 4; axNo++)
                    {
                        SendString("AXI" + axNo.ToString() + ":UNIT 0:SELSP 0");
                        System.Threading.Thread.Sleep(100);
                    }

                    ButtonAxisX.Enabled = true;
                    ButtonAxisY.Enabled = true;
                    ButtonAxisZ.Enabled = true;
                    ButtonAxisU.Enabled = true;
                }
                //Number of axes is 6(X-W)
                else
                {
                    //---------------------------------------------------------
                    // Setting unit and speed table
                    //---------------------------------------------------------
                    for (int axNo = 1; axNo <= 6; axNo++)
                    {
                        SendString("AXI" + axNo.ToString() + ":UNIT 0:SELSP 0");
                        System.Threading.Thread.Sleep(100);
                    }

                    ButtonAxisX.Enabled = true;
                    ButtonAxisY.Enabled = true;
                    ButtonAxisZ.Enabled = true;
                    ButtonAxisU.Enabled = true;
                    ButtonAxisV.Enabled = true;
                    ButtonAxisW.Enabled = true;
                }

                UpdateStatus();
            }
            else
            {
                LabelState.Text = "Receive error";
                MessageBox.Show("DS102/DS112 controller is not connected to " + ComboBoxCommPort.Text);
            }
        }

        ///<summary>
        ///When press the disconnect button
        ///</summary>
        private void ButtonDisconnect_Click(object sender, EventArgs e)
        {
            if (SerialPort != null)
                SerialPort.Close();

            ButtonAxisX.Enabled = true;
            ButtonAxisY.Enabled = true;
            ButtonAxisZ.Enabled = true;
            ButtonAxisU.Enabled = true;
            ButtonAxisV.Enabled = true;
            ButtonAxisW.Enabled = true;

            LabelState.Text = "Disconnected";
        }

        ///<summary>
        ///When select the X button
        ///</summary>
        private void ButtonAxisX_CheckedChanged(object sender, EventArgs e)
        {
            AxisNo = "1";           //Set the active axis to the X-axis
            UpdateStatus();
        }

        ///<summary>
        ///When select the Y button
        ///</summary>
        private void ButtonAxisY_CheckedChanged(object sender, EventArgs e)
        {
            AxisNo = "2";           //Set the active axis to the Y-axis
            UpdateStatus();
        }

        ///<summary>
        ///When select the Z button
        ///</summary>
        private void ButtonAxisZ_CheckedChanged(object sender, EventArgs e)
        {
            AxisNo = "3";           //Set the active axis to the Z-axis
            UpdateStatus();
        }

        ///<summary>
        ///When select the U button
        ///</summary>
        private void ButtonAxisU_CheckedChanged(object sender, EventArgs e)
        {
            AxisNo = "4";           //Set the active axis to the U-axis
            UpdateStatus();
        }

        ///<summary>
        ///When select the V button
        ///</summary>
        private void ButtonAxisV_CheckedChanged(object sender, EventArgs e)
        {
            AxisNo = "5";           //Set the active axis to the V-axis
            UpdateStatus();
        }

        ///<summary>
        ///When select the W button
        ///</summary>
        private void ButtonAxisW_CheckedChanged(object sender, EventArgs e)
        {
            AxisNo = "6";           //Set the active axis to the W-axis
            UpdateStatus();
        }

        ///<summary>
        ///When select the continue button
        ///</summary>
        private void RadioButtonContinueMode_CheckedChanged(object sender, EventArgs e)
        {
            ButtonCCW.Text = "- (CCW)";
            ButtonCW.Text = "+ (CW)";
            Mode = 0;
        }

        ///<summary>
        ///When select the step button
        ///</summary>
        private void RadioButtonStepMode_CheckedChanged(object sender, EventArgs e)
        {
            ButtonCCW.Text = "- (CCW)";
            ButtonCW.Text = "+ (CW)";
            Mode = 1;
        }

        ///<summary>
        ///When select the origin button
        ///</summary>
        private void RadioButtonOrgMode_CheckedChanged(object sender, EventArgs e)
        {
            ButtonCCW.Text = "Origin";
            ButtonCW.Text = "Origin";
            Mode = 2;
        }

        ///<summary>
        ///When press the stop button
        ///</summary>
        private void ButtonStop_Click(object sender, EventArgs e)
        {
            //---------------------------------------------------------
            // Stop
            //---------------------------------------------------------
            SendString("STOP 0");
        }

        ///<summary>
        ///When press the CCW button
        ///</summary>
        private void ButtonCCW_MouseDown(object sender, MouseEventArgs e)
        {
            Direction = "CCW";              //Set the motor drive direction to CCW
            MoveStage();                    //Start the drive
        }

        ///<summary>
        ///When release the CCW button
        ///</summary>
        private void ButtonCCW_MouseUp(object sender, MouseEventArgs e)
        {
            //In the case of continue drive mode
            if (Mode == 0)
                //---------------------------------------------------------
                // Stop
                //---------------------------------------------------------
                SendString("STOP 0");       //Stop the axis
        }

        ///<summary>
        ///When press the CW button
        ///</summary>
        private void ButtonCW_MouseDown(object sender, MouseEventArgs e)
        {
            Direction = "CW";               //Set the motor drive direction to CW
            MoveStage();                    //Start the drive
        }

        ///<summary>
        ///When release the CW button
        ///</summary>
        private void ButtonCW_MouseUp(object sender, MouseEventArgs e)
        {
            //In the case of continue drive mode
            if (Mode == 0)
                //---------------------------------------------------------
                // Stop
                //---------------------------------------------------------
                SendString("STOP 0");       //Stop the axis
        }

        ///<summary>
        ///Drive the stage
        ///</summary>
        private void MoveStage()
        {
            //Drive mode
            switch (Mode)
            {
                //Continue
                case 0:
                    //If the mortor drive direction is CW
                    if (Direction == "CW")
                        //---------------------------------------------------------
                        // Continue drive / CW direction
                        //---------------------------------------------------------
                        SendString("AXI" + AxisNo + ":L0 " + TextBoxLSpeed.Text +
                                   ":R0 " + TextBoxRate.Text + ":S0 " + TextBoxSRate.Text +
                                   ":F0 " + TextBoxSpeed.Text + ":GO CWJ");
                    //If the mortor drive direction is CCW
                    else
                        //---------------------------------------------------------
                        // Continue drive / CCW direction
                        //---------------------------------------------------------
                        SendString("AXI" + AxisNo + ":L0 " + TextBoxLSpeed.Text +
                                   ":R0 " + TextBoxRate.Text + ":S0 " + TextBoxSRate.Text +
                                   ":F0 " + TextBoxSpeed.Text + ":GO CCWJ");
                    break;
                //Step
                case 1:
                    //---------------------------------------------------------
                    // Step drive
                    //---------------------------------------------------------
                    SendString("AXI" + AxisNo + ":L0 " + TextBoxLSpeed.Text +
                              ":R0 " + TextBoxRate.Text + ":S0 " + TextBoxSRate.Text +
                              ":F0 " + TextBoxSpeed.Text + ":PULS " + TextBoxStep.Text + ":GO " + Direction);
                    break;
                //Origin
                case 2:
                    //Get the origin type
                    string orgMode = ComboBoxOrg.SelectedIndex.ToString();

                    //Change the origin type
                    //---------------------------------------------------------
                    // Memory switch 0 setting
                    //---------------------------------------------------------
                    SendString("AXI" + AxisNo + ":MEMSW0 " + orgMode);

                    System.Threading.Thread.Sleep(100);

                    //---------------------------------------------------------
                    // Origin drive
                    //---------------------------------------------------------
                    SendString("AXI" + AxisNo + ":L0 " + TextBoxLSpeed.Text +
                               ":R0 " + TextBoxRate.Text + ":S0 " + TextBoxSRate.Text +
                               ":F0 " + TextBoxSpeed.Text + ":GO ORG");
                    break;
            }

            Timer.Start();
        }

        ///<summary>
        ///Screen update
        ///</summary>
        private void Timer_Tick(object sender, EventArgs e)
        {
            if (SerialPort.IsOpen)
                UpdateStatus();
        }

        ///<summary>
        ///Status update
        ///</summary>
        private void UpdateStatus()
        {
            string data;

            //---------------------------------------------------------
            // Request status binary 3
            //---------------------------------------------------------
            data = SendReceiveString("AXI" + AxisNo + ":SB3?");

            //Terminates if it cannot be converted to a numerical value
            if (!int.TryParse(data, out int result))
            {
                Timer.Stop();
                return;
            }

            //If axis selection is not available
            if ((int.Parse(data) & 0x01) != 0x01)
            {
                LabelState.Text = "Axes cannot be selected";
                Timer.Stop();
            }
            else
            {
                //---------------------------------------------------------
                // Request status binary 1
                //---------------------------------------------------------
                data = SendReceiveString("AXI" + AxisNo + ":SB1?");

                if ((int.Parse(data) & 0x40) == 0x40)
                {
                    LabelState.Text = "Driving";
                }
                else if ((int.Parse(data) & 0x10) == 0x10)
                {
                    LabelState.Text = "Detect origin";
                    Timer.Stop();
                }
                else if ((int.Parse(data) & 0x02) == 0x02 || (int.Parse(data) & 0x04) == 0x04)
                {
                    //Detect limit
                    //---------------------------------------------------------
                    // Request status binary 2
                    //---------------------------------------------------------
                    data = SendReceiveString("AXI" + AxisNo + ":SB2?");

                    if ((int.Parse(data) & 0x03) == 0x03)
                    {
                        LabelState.Text = "Stage not connected";
                    }
                    else if ((int.Parse(data) & 0x01) == 0x01)
                    {
                        LabelState.Text = "Detect CW limit";
                    }
                    else if ((int.Parse(data) & 0x02) == 0x02)
                    {
                        LabelState.Text = "Detect CCW limit";
                    }
                    else if ((int.Parse(data) & 0x04) == 0x04)
                    {
                        LabelState.Text = "Detect CW software limit";
                    }
                    else if ((int.Parse(data) & 0x08) == 0x08)
                    {
                        LabelState.Text = "Detect CCW software limit";
                    }

                    Timer.Stop();
                }
                else
                {
                    LabelState.Text = "Stop";
                    Timer.Stop();
                }

                //---------------------------------------------------------
                // Request the current position
                //---------------------------------------------------------
                data = SendReceiveString("AXI" + AxisNo + ":POS?");
                TextBoxPosition.Text = data;
            }
        }

        ///<summary>
        ///Send
        ///</summary>
        ///<param name="cmd">Send string</param>
        public void SendString(string cmd)
        {
            try
            {
                if (SerialPort.IsOpen)
                    //Send
                    SerialPort.WriteLine(cmd);
            }
            catch (Exception ex)
            {
                LabelState.Text = "Send error";
                MessageBox.Show(ex.Message);
            }
        }

        ///<summary>
        ///Receive
        ///</summary>
        ///<returns>Receive string</returns>
        public string ReceiveString()
        {
            DateTime timeoutTime = DateTime.Now.AddMilliseconds(2000);      //Timeout time
            string rcvData = null;                                          //Receive data

            //Perform receiving process until the delimiter code
            while (true)
            {
                try
                {
                    int len = SerialPort.BytesToRead;

                    //Attempt receiving process until timeout
                    while (len < 1)
                    {
                        System.Threading.Thread.Sleep(50);

                        if (DateTime.Now > timeoutTime)
                        {
                            LabelState.Text = "Timeout error";
                            return null;
                        }

                        len = SerialPort.BytesToRead;
                    }

                    int readBytes = 0;
                    byte[] inByte = new byte[len];

                    //Read data from the received buffer
                    SerialPort.Read(inByte, readBytes, len);
                    //Store the the received data in a string
                    string rcvDataTemp = Encoding.ASCII.GetString(inByte);

                    rcvData += rcvDataTemp;

                    //Confirm the position of the delimiter code
                    int index = rcvData.IndexOf(SerialPort.NewLine);

                    //When a delimiter code is found
                    if (index > -1)
                    {
                        //Remove the delimiter code
                        string rcvDataReturn = rcvData.Remove(index);

                        return rcvDataReturn;
                    }
                }
                catch (Exception ex)
                {
                    LabelState.Text = "Receive error";
                    MessageBox.Show(ex.Message);
                    return null;
                }
            }
        }

        ///<summary>
        ///Send and receive
        ///</summary>
        ///<param name="cmd">Send string</param>
        ///<returns>Receive string</returns>
        public string SendReceiveString(string cmd)
        {
            string data = null;

            if (SerialPort.IsOpen)
            {
                //Send
                SendString(cmd);

                System.Threading.Thread.Sleep(100);

                //Receive
                data = ReceiveString();
            }

            return data;
        }


        ///<summary>
        ///When press the position set button
        ///</summary>
        private void ButtonSetPosition_Click(object sender, EventArgs e)
        {
            //---------------------------------------------------------
            // Set the current position
            //---------------------------------------------------------
            SendString("AXI" + AxisNo + ":POS " + TextBoxPosition.Text);
        }

        ///<summary>
        ///When press the close button
        ///</summary>
        private void ButtonClose_Click(object sender, EventArgs e)
        {
            Close();
        }

        ///<summary>
        ///End-of-process handling
        ///</summary>
        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (SerialPort.IsOpen)
                SerialPort.Close();
        }
    }
}